#include <allegro.h>
#include <stdio.h>

#include "main.h"
#include "game.h"
#include "pbox.h"
#include "gfx.h"
#include "text.h"
#include "map.h"
#include "lord.h"
#include "chatbox.h"

volatile int lord_count;
LORD lords[5];
int castle_count;

int game_mode;

/* number of people still moving */
int still_moving = 0;
/* this gets set when a slave is OK to move */
int i_can_move = FALSE;
/* the id of the slave */
int who_i_am = 0; 
/* this also serves as an abort flag for people spending too much time on their moves */
int time_to_watch = FALSE;

int lost_connection = FALSE;
BITMAP *activity_bmp = NULL;
GFX_OVERLAY *activity_overlay = NULL;

/* watch modes, ouch this is bad */
#define WATCH_MOVING          0
#define WATCH_FIGHTING        1
#define WATCH_END_OF_FIGHT    2 
typedef struct WATCH_STATUS
{
	int mode;
	int x, y;
	int attacker_id, defender_id;
	int attacker_size, defender_size;
} WATCH_STATUS;
WATCH_STATUS watch_status;


BITMAP *attack_sprite[3];

/* a hack to make up for the fact that the readkey()s in the original hot seat mode were bad for enet */
int game_readkey()
{
	while (!keypressed())
	{
		game_poll();
		if (game_mode == MODE_SLAVE && time_to_watch) return 0;
	}

	return readkey();
}

void game_pause()
{
	clear_keybuf();
	if (game_mode == MODE_HOTSEAT)
		readkey();
	else
	{
		ticks = 0;
		while (!keypressed() && ticks < 90)
		{
			game_poll();
		}

		if (keypressed()) readkey();
	}
}

void game_activity_display_init()
{
	activity_bmp = create_bitmap(8, 32);
	clear(activity_bmp);
	activity_overlay = gfx_add_overlay(gfx, activity_bmp, 312,8, 1);
}

void game_activity_display_shutdown()
{
	if (activity_bmp)
	{
		destroy_bitmap(activity_bmp);
		activity_bmp = NULL;
	}
	gfx_remove_overlay(gfx, activity_overlay);	
}

void game_init()
{
	int i;
	strcpy(lords[0].name, "REBEL");
	attack_sprite[0] = load_bitmap("data/attack1.pcx", NULL);
	attack_sprite[1] = load_bitmap("data/attack2.pcx", NULL);
	attack_sprite[2] = load_bitmap("data/attack3.pcx", NULL);

	for (i=0; i<=4; i++)
	{
		lords[i].last_activity = 0;
		lords[i].done_with_turn = FALSE;
	}
}

void game_shutdown()
{
	destroy_bitmap(attack_sprite[0]);
	destroy_bitmap(attack_sprite[1]);
	destroy_bitmap(attack_sprite[2]);
}

void game_end()
{
	map_destroy(map);
	map = NULL;
	if (game_mode != MODE_HOTSEAT)
	{
		chatbox_shutdown();
		game_activity_display_shutdown();
	}
}

void game_begin()
{
	BITMAP *buffer;
	int i, level;
	int y = 0;
	char response[25];

	game_mode = MODE_HOTSEAT;
	
	buffer = create_bitmap(gfx->buffer->w, gfx->buffer->h);

	lord_count = prompt_int(NULL, 1,4, 0,0, 15, -1, "LORDS (1-4)?");
	blit(gfx->buffer, buffer, 0,0, 0,0, buffer->w, buffer->h);

	lords[0].is_dead = FALSE;
	for (i=1; i<=lord_count; i++)
	{
		y += 8;
		lords[i].name[0] = 0;
		prompt(buffer, lords[i].name, 15, 0, y, 15, -1, "LORD %d'S NAME?", i);
		lords[i].is_dead = FALSE;
		blit(gfx->buffer, buffer, 0,0, 0,0, buffer->w, buffer->h);
	}

	y += 8;
	castle_count = prompt_int(buffer, 10,35, 0,y, 15, -1, "CASTLES (10-35)?");
	blit(gfx->buffer, buffer, 0,0, 0,0, buffer->w, buffer->h);

	y += 8;	textprintf_ex(buffer, font64, 0, y, 15, -1, "LEVELS: 1-NO TERRAIN");
	y += 8; textprintf_ex(buffer, font64, 0, y, 15, -1, "        2-HILLS, SOME TREES");
	y += 8; textprintf_ex(buffer, font64, 0, y, 15, -1, "        3-HILLS AND FORESTS");
	y += 8;

	level = prompt_int(buffer, 1,3, 0,y, 15, -1, "LEVEL (1-3)?");
	

	map = NULL;
	do {
		if (map) map_destroy(map);
		map = map_create(level);
		map_add_castles(map, lord_count, castle_count);

		gfx_frame_start(gfx);		
		map_draw(map, gfx->buffer, 8,8);
		gfx_frame_end(gfx);
		blit(gfx->buffer, buffer, 0,0, 0,0, buffer->w, buffer->h);
		response[0] = 0;
		prompt(buffer, response, 2, 8,20 * 8, 1,-1, "OK (Y/N)?");
	} while (response[0] != 'Y');

	destroy_bitmap(buffer);
}

/* Master gives slaves something to read */
void game_master_broadcast_chat(int player, char *message)
{
	PACKET_BOX *pb = packet_box_create("TALK");
	ENetPacket *packet;
	packet_box_add_int(pb, player, 2);
	packet_box_add_string(pb, message);
	packet_box_serialize(pb);

	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);

	packet_box_destroy(pb);

	chatbox_add(player, message);
}

/* send an activity packet so that people know who's still around and kicking */
void game_master_broadcast_pong(int player)
{
	PACKET_BOX *pb = packet_box_create("PONG");
	ENetPacket *packet;
	packet_box_add_int(pb, player, 2);
	packet_box_serialize(pb);

	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);

	packet_box_destroy(pb);
}

/* tell everyone that someone is a quitter */
void game_master_broadcast_quitter(int player)
{
	PACKET_BOX *pb = packet_box_create("QUIT");
	ENetPacket *packet;
	packet_box_add_int(pb, player, 2);
	packet_box_serialize(pb);

	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);

	packet_box_destroy(pb);
}

/* tell everyone that someone is done with his turn */
void game_master_broadcast_done(int player)
{
	PACKET_BOX *pb = packet_box_create("DONE");
	ENetPacket *packet;
	packet_box_add_int(pb, player, 2);
	packet_box_serialize(pb);

	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);

	packet_box_destroy(pb);
}


/*
	Master tells slaves how many soldiers are in the castle.
	If updated_castle is NULL, then send all.
*/
void game_send_soldier_count(CASTLE *updated_castle)
{
	int i;

	for (i=2; i<=lord_count; i++)
	{
		CASTLE *castle = updated_castle ? updated_castle : map->castles;
		PACKET_BOX *pb = packet_box_create("SLDR");
		ENetPacket *packet;

		if (castle)
		{
			do 
			{
				packet_box_add_int(pb, castle->player, 2);
				packet_box_add_int(pb, castle->x, 8);
				packet_box_add_int(pb, castle->y, 8);

				if (castle->player == i)
				{
					/* Home sweet home. */
					packet_box_add_int(pb, castle->soldiers, 32);
					packet_box_add_int(pb, castle->levy, 8);
				}
				else
				{
					/* Not your castle? No troop info for you. */
					packet_box_add_int(pb, 0, 0);
					packet_box_add_int(pb, 0, 0);
				}
				castle = castle->next;
			} while (castle && !updated_castle);
		}
		packet_box_add_int(pb, 99, 8); /* 99 = end of sequence marker */
		packet_box_serialize(pb);

		packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
		enet_peer_send(peer, (unsigned char) i, packet);

		packet_box_destroy(pb);
	}
}

/* Unpacks a map as received from the host */
void game_process_net_map(PACKET_BOX *pb)
{
	int castle_count;
	int x,y;
	int v[] = {TILE_GRASS, TILE_TREES, TILE_ROCKS, TILE_CASTLE };

	map = map_create(0);

	for (y=0; y<19; y++)
	{
		unsigned int d = packet_box_get_int(pb);

		map->grid[y][0].tile = v[(d >> 30) & 3];
		map->grid[y][1].tile = v[(d >> 28) & 3];
		map->grid[y][2].tile = v[(d >> 26) & 3];
		map->grid[y][3].tile = v[(d >> 24) & 3];
		map->grid[y][4].tile = v[(d >> 22) & 3];
		map->grid[y][5].tile = v[(d >> 20) & 3];
		map->grid[y][6].tile = v[(d >> 18) & 3];
		map->grid[y][7].tile = v[(d >> 16) & 3];
		map->grid[y][8].tile = v[(d >> 14) & 3];
		map->grid[y][9].tile = v[(d >> 12) & 3];
		map->grid[y][10].tile = v[(d >> 10) & 3];
		map->grid[y][11].tile = v[(d >> 8) & 3];
		map->grid[y][12].tile = v[(d >> 6) & 3];
		map->grid[y][13].tile = v[(d >> 4) & 3];
		map->grid[y][14].tile = v[(d >> 2) & 3];
		map->grid[y][15].tile = v[d & 3];

		d = packet_box_get_int(pb);
		map->grid[y][16].tile = v[(d >> 4) & 3];
		map->grid[y][17].tile = v[(d >> 2) & 3];
		map->grid[y][18].tile = v[d & 3];
	}

	for (y=0; y<19; y++)
		for (x=0; x<19; x++)
		{
			map->grid[y][x].castle = NULL;
			map->grid[y][x].x = x;
			map->grid[y][x].y = y;
		}

	castle_count = 	packet_box_get_int(pb);

	while (castle_count--)
	{
		CASTLE *castle = malloc(sizeof(CASTLE));
		castle->levy = -1;
		castle->player = packet_box_get_int(pb);
		castle->soldiers = -1;
		castle->x = packet_box_get_int(pb);
		castle->y = packet_box_get_int(pb);

		map->grid[castle->y][castle->x].castle = castle;
		
		castle->next = map->castles;
		map->castles = castle;
	}
}

void game_process_net_move(int player_num, PACKET_BOX *pb)
{
	int sx = packet_box_get_int(pb);
	int sy = packet_box_get_int(pb);
	int dx = packet_box_get_int(pb);
	int dy = packet_box_get_int(pb);
	int size = packet_box_get_int(pb);


	ARMY *army = army_add(&map->armies);
	CASTLE *castle = map->grid[sy][sx].castle; 
	if (!castle)
	{
		ARMY *idle_army = army_get_idle(map->armies, sx,sy);
		
		if (idle_army->size < size) size = idle_army->size; /* cheaters */
			idle_army->size -= size;

		if (idle_army->size == 0)
			army_remove(&map->armies, idle_army);
		
	}
	else
	{
		if (castle->soldiers == 0) return; /* cheaters */
		if (castle->soldiers < size) size = castle->soldiers; /* cheaters */
		castle->soldiers -= size;
	}

	army->player = player_num;
	army->x = sx;
	army->y = sy;
	army->dx = dx;
	army->dy = dy;
	army->size = size;
	army->idle = FALSE;
	army->moves_left = 5;

}

int game_poll_enet()
{
	ENetEvent event;

	if (game_mode == MODE_HOTSEAT)
	{
		/* FIX ME: we shouldn't be here, whoops!!! */
		rest(10);
		return 0;
	}

	while (enet_host_service (client, & event, 10) > 0)
	{
		if (event.type == ENET_EVENT_TYPE_DISCONNECT)
		{			
			lost_connection = TRUE;
		}
		else if (event.type == ENET_EVENT_TYPE_RECEIVE)
		{
			PACKET_BOX *pb = packet_box_unserialize(event.packet->data, event.packet->dataLength);
			enet_packet_destroy(event.packet);

			printf("Received PacketBox [%p]: %s\n", pb, pb->cmd);

			if (event.channelID == 0)
			{
				if (!strncmp(pb->cmd, "WHO ", 4))
				{
					int i;
					who_i_am = packet_box_get_int(pb);
					lord_count = packet_box_get_int(pb);
					for (i=0; i<lord_count; i++)
						packet_box_get_string(pb, lords[i+1].name, 80);
				}
			}
			else if (event.channelID == 1)
			{
				if (game_mode == MODE_SLAVE)
				{
					lords[1].last_activity = time(NULL);

					if (!strncmp(pb->cmd, "MAP ", 4))
					{
						game_process_net_map(pb);
					}
					else if (!strncmp(pb->cmd, "MOVE", 4))
					{
						int i;
						for (i=1; i<=lord_count; i++)
							lords[i].done_with_turn = FALSE;
						i_can_move = TRUE;
						time_to_watch = FALSE;
					}
					else if (!strncmp(pb->cmd, "PONG", 4))
					{
						int player = packet_box_get_int(pb);
						lords[player].last_activity = time(NULL);
					}
					else if (!strncmp(pb->cmd, "WTCH", 4))
					{
						time_to_watch = TRUE;
						watch_status.mode = WATCH_MOVING;
					}
					else if (!strncmp(pb->cmd, "ADED", 4))
					{
						ARMY *army = map->armies;
						int id = packet_box_get_int(pb);
						while (army)
						{
							if (army->id == id)
							{
								army_remove(&map->armies, army);
								break;
							}
							army = army->next;
						}
					}
					else if (!strncmp(pb->cmd, "FITE", 4))
					{
						static int pan = 0;

						watch_status.mode = WATCH_FIGHTING;
						watch_status.x = packet_box_get_int(pb);
						watch_status.y = packet_box_get_int(pb);
						watch_status.attacker_id = packet_box_get_int(pb);
						watch_status.attacker_size = packet_box_get_int(pb);
						watch_status.defender_id = packet_box_get_int(pb);
						watch_status.defender_size = packet_box_get_int(pb);

						if (watch_status.attacker_size && watch_status.defender_size)
							play_sample(poof, 140, pan = 255-pan, 1000, 0);

					}
					else if (!strncmp(pb->cmd, "FITO", 4))
					{
						watch_status.mode = WATCH_END_OF_FIGHT;
						watch_status.defender_size = packet_box_get_int(pb);
						if (!watch_status.defender_size)
						{
							play_sample(ahahaa, 50, 128, 1000, 0);
						}
					}
					else if (!strncmp(pb->cmd, "ARST", 4))
					{
						army_destroy(&map->armies);
					}
					else if (!strncmp(pb->cmd, "ARMY", 4))
					{
						int id = packet_box_get_int(pb);
						while (id)
						{
							int x = packet_box_get_int(pb);
							int y = packet_box_get_int(pb);
							int player = packet_box_get_int(pb);
							int size = packet_box_get_int(pb);
							int idle = packet_box_get_int(pb);

							ARMY *army = map->armies;
							while (army)
							{
								if (army->id == id) break;
								army = army->next;
							}
	
							if (!army)
							{
								army = army_add(&map->armies);
								army->id = id;								
							}

							army->x = x;
							army->y = y;
							army->size = size;
							army->idle = idle;
							army->player = player;

							id = packet_box_get_int(pb);
						}
					}
					else if (!strncmp(pb->cmd, "SLDR", 4))
					{
						int player = packet_box_get_int(pb);

						while (player != 99)
						{
							int x = packet_box_get_int(pb);
							int y = packet_box_get_int(pb);
							map->grid[y][x].castle->player = player;
							map->grid[y][x].castle->soldiers = packet_box_get_int(pb);						 
							map->grid[y][x].castle->levy = packet_box_get_int(pb);

							player = packet_box_get_int(pb);
						}
					}
					else if (!strncmp(pb->cmd, "TALK", 4))
					{
						char message[40];
						int player = packet_box_get_int(pb);
						packet_box_get_string(pb, message, 40);
						if (player != who_i_am) chatbox_add(player, message);
					}
					else if (!strncmp(pb->cmd, "DONE", 4))
					{
						int player = packet_box_get_int(pb);						
						lords[player].done_with_turn = TRUE;
					}
					else if (!strncmp(pb->cmd, "QUIT", 4))
					{
						int player = packet_box_get_int(pb);						
						lords[player].is_dead = TRUE;
					}
				}
			}
			else if (event.channelID >= 2 && event.channelID <= lord_count)
			{
				if (game_mode == MODE_MASTER)
				{
					game_master_broadcast_pong(event.channelID);

					if (!strncmp(pb->cmd, "ASGN", 4))
					{
						game_process_net_move(event.channelID, pb);
					}
					else if (!strncmp(pb->cmd, "QUIT", 4))
					{
						/* we really don't care if someone quit after they died */
						if (lords[event.channelID].is_dead == FALSE)
						{
							lords[event.channelID].is_dead = TRUE;
							if (!lords[event.channelID].done_with_turn)
							{
								lords[event.channelID].done_with_turn = TRUE;
								still_moving--;
							}
							game_master_broadcast_quitter(event.channelID);
						}
					}
					else if (!strncmp(pb->cmd, "DONE", 4))
					{						
						if (!lords[event.channelID].done_with_turn && !lords[event.channelID].is_dead)
						{
							/* don't count them twice, and no dead voters please */
							lords[event.channelID].done_with_turn = TRUE;
							game_master_broadcast_done(event.channelID);
							still_moving--;							
						}
					}
					else if (!strncmp(pb->cmd, "TALK", 4))
					{
						char message[40];
						packet_box_get_string(pb, message, 40);
						game_master_broadcast_chat(event.channelID, message);
					}
				}
			}

			packet_box_destroy(pb);
		}
	}

	return 0;
}


void game_begin_network(int castle_count, int level)
{
	int y;
	PACKET_BOX *pb;
	ENetPacket *packet;
	CASTLE *castle;
	int v[256];

	game_activity_display_init();

	game_mode = MODE_MASTER;
	lord_count = 0;

	pb = packet_box_create("WHO ");
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 0, packet);
	packet_box_destroy(pb);
	
	while (!key[KEY_ESC] && !lord_count) game_poll();

	if (!lord_count) exit(1);
	
	map = map_create(level);
	map_add_castles(map, lord_count, castle_count);

	/*
		each tile is two bits (binary: 00 01 10 11)
		using 32 bit integers means , 16 tiles per integer
		i'm not going to bother packing the second int very tightly here....

	    |                 FIRST INT                   | |  SECOND INT ....... |
		00 01 02 03 04 05 06 07 08 09 10 11 12 13 14 15 16 17 18
	*/

	v[TILE_GRASS] = 0;
	v[TILE_TREES] = 1;
	v[TILE_ROCKS] = 2;
	v[TILE_CASTLE] = 3;
	
	pb = packet_box_create("MAP ");

	for (y=0; y<19; y++)
	{
		int d1 = 		
			((v[map->grid[y][0].tile]&3) << 30) |
			((v[map->grid[y][1].tile]&3) << 28) |
			((v[map->grid[y][2].tile]&3) << 26) |
			((v[map->grid[y][3].tile]&3) << 24) |
			((v[map->grid[y][4].tile]&3) << 22) |
			((v[map->grid[y][5].tile]&3) << 20) |
			((v[map->grid[y][6].tile]&3) << 18) |
			((v[map->grid[y][7].tile]&3) << 16) |
			((v[map->grid[y][8].tile]&3) << 14) |
			((v[map->grid[y][9].tile]&3) << 12) |
			((v[map->grid[y][10].tile]&3) << 10) |
			((v[map->grid[y][11].tile]&3) << 8) |
			((v[map->grid[y][12].tile]&3) << 6) |
			((v[map->grid[y][13].tile]&3) << 4) |
			((v[map->grid[y][14].tile]&3) << 2) |
			(v[map->grid[y][15].tile]&3)
		;

		packet_box_add_int(pb, d1, 32);

		d1 = 		
			((v[map->grid[y][16].tile]&3) << 4) |
			((v[map->grid[y][17].tile]&3) << 2) |
			(v[map->grid[y][18].tile]&3)
		;

		packet_box_add_int(pb, d1, 32);
	}

	packet_box_add_int (pb, castle_count, 8);

	castle = map->castles;
	while (castle)
	{		
		packet_box_add_int(pb, castle->player, 8);
		packet_box_add_int(pb, castle->x, 8);
		packet_box_add_int(pb, castle->y, 8);
		castle = castle->next;
	}


	packet_box_serialize(pb);

	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);

	packet_box_destroy(pb);

	chatbox_init();
}

/* Master tells slaves, "Army went here." */
void game_send_army_position(ARMY *army)
{	
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("ARMY");

	packet_box_add_int(pb, army->id, 8);
	packet_box_add_int(pb, army->x, 8);
	packet_box_add_int(pb, army->y, 8);
	packet_box_add_int(pb, army->player, 8);	
	packet_box_add_int(pb, 0, 8); /* Size is a secret */
	packet_box_add_int(pb, 0, 8); /* Idle is a secret */

	packet_box_add_int(pb, 0, 0); /* EOF */

	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);
	packet_box_destroy(pb);
}

/* Master tells slaves, "We have a fight." */
void game_send_attack_signal(ARMY *army, int enemy, int enemy_size)
{
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("FITE");
	packet_box_add_int(pb, army->x, 8);
	packet_box_add_int(pb, army->y, 8);
	packet_box_add_int(pb, army->player, 2);
	packet_box_add_int(pb, army->size, 32);
	packet_box_add_int(pb, enemy, 2);
	packet_box_add_int(pb, enemy_size, 32);
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);
	packet_box_destroy(pb);
}

/* Master tells slaves, "We're all done fighting.." */
void game_send_attack_over(int enemy_size)
{
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("FITO");
	packet_box_add_int(pb, enemy_size, 32);
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);
	packet_box_destroy(pb);
}

/* Master tells slaves, "This army is no more." */
void game_send_army_dead(ARMY *army)
{	
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("ADED");
	packet_box_add_int(pb, army->id, 8);
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);
	packet_box_destroy(pb);
}

/* Master tells slaves, "Here's how big your armies are." */
void game_send_army_count()
{
	int i;
	for (i=2; i<=lord_count; i++)
	{
		ARMY *army = map->armies;
		PACKET_BOX *pb = packet_box_create("ARST"); /* Reset Army List */
		ENetPacket *packet;
		packet_box_serialize(pb);
		packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
		enet_peer_send(peer, (unsigned char) i, packet);
		packet_box_destroy(pb);

		pb = packet_box_create("ARMY");

		while (army)
		{
			packet_box_add_int(pb, army->id, 8);
			if (army->player == i || map->grid[army->y][army->x].tile == TILE_GRASS)
			{
				packet_box_add_int(pb, army->x, 8);
				packet_box_add_int(pb, army->y, 8);								
			}
			else
			{
				packet_box_add_int(pb, 20, 8);
				packet_box_add_int(pb, 20, 8);
			}
			packet_box_add_int(pb, army->player, 8);

			if (army->player == i)
			{
				packet_box_add_int(pb, army->size, 32);				
				packet_box_add_int(pb, army->idle, 32);
			}
			else
			{
				packet_box_add_int(pb, 0, 0);
				packet_box_add_int(pb, 0, 0);
			}
			
			army = army->next;
		}
		packet_box_add_int(pb, 0, 0); /*EOF */
		packet_box_serialize(pb);

		packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
		enet_peer_send(peer, (unsigned char) i, packet);

		packet_box_destroy(pb);
	}
}

void game_begin_client()
{
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("WHO ");
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 0, packet);
	packet_box_destroy(pb);

	game_activity_display_init();	

	map = 0;
	game_mode = MODE_SLAVE;
	who_i_am = lord_count = 0;
	
	while (!key[KEY_ESC] && (!lord_count || !who_i_am)) game_poll();
	if (!lord_count) exit(1);

	while (!key[KEY_ESC] && !map) game_poll();
	if (!map) exit(1);

	chatbox_init();
}

/* Master tells Slaves, "Move!" */
void game_send_start_signal()
{
	int i;
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("MOVE");
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);
	packet_box_destroy(pb);

	time_to_watch = FALSE;
	still_moving = 0;
	for (i=1; i<=lord_count; i++)
	{
		lords[i].done_with_turn = FALSE;
		if (lords[i].is_dead == FALSE) still_moving++;
	}
}

/* Master tells Slaves, "Watch Me." */
void game_send_watch_signal()
{
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("WTCH");
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);
	packet_box_destroy(pb);
}

/* Slave tells Master, "I'm Done!" */
void game_send_done_with_turn_signal()
{
	ENetPacket *packet;	
	PACKET_BOX *pb = packet_box_create("DONE");
	packet_box_serialize(pb);
	packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
	enet_peer_send(peer, 1, packet);
	packet_box_destroy(pb);

	lords[who_i_am].done_with_turn = TRUE;
}

void game_type_chat_message(int player, unsigned char first_char)
{
	char text[40];
	BITMAP *buffer = create_bitmap(320,200);
	blit(gfx->buffer, buffer, 0,0, 0,0, 320,200);
	rectfill(buffer, 0,160, 319,199, 0);
	textprintf_ex(buffer, font64, 0,160, 15,0, "SEND NET MESSAGE:");
	
	if (first_char >= 'a' && first_char <= 'z') first_char += ('A' - 'a');

	text[0] = first_char;
	text[1] = 0;
	prompt(buffer, text, 39, 0,168, 4+player,0, "");
	destroy_bitmap(buffer);

	if (text[0] != 0)
	{
		if (game_mode == MODE_MASTER)
		{
			game_master_broadcast_chat(1, text);
		}
		else
		{
			ENetPacket *packet;	
			PACKET_BOX *pb = packet_box_create("TALK");
			packet_box_add_string(pb, text);
			packet_box_serialize(pb);
			packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
			enet_peer_send(peer, 1, packet);
			packet_box_destroy(pb);

			chatbox_add(player, text);
		}
	}
}

#define MODE_LOOKING       0
#define MODE_CREATE_ARMY   1
#define MODE_SENDING       2
#define MODE_SEND_CONFIRM  3

int game_take_turn(int i)
{
	BITMAP *buffer;

	CASTLE *castle;
	int x=9, y=9, sx=0,sy=0;
	int done = 0;
	int mode = MODE_LOOKING;
	int sending = 0, max_soldiers = 0;
	int no_troops_msg = 0;
	int army_is_here = -1;
	int is_dead = TRUE;

	if (i == 0 || lords[i].is_dead)
		return 0;

	castle = map->castles;
	while (castle)
	{
		if (castle->player == i)
		{
			is_dead = FALSE;
			break;
		}
		castle = castle->next;
	}

	if (is_dead)
	{
		ARMY *army = map->armies;
		while (army)
		{
			if (army->player == i)
			{
				is_dead = FALSE;
				break;
			}
			army = army->next;
		}
	}

	if (is_dead)
	{
		buffer = gfx_frame_start(gfx);
		map_draw(map, buffer, 8,8);
		textprintf_ex(buffer, font64, 8,0, 4+i,0, "%s DID NOT SURVIVE!", lords[i].name);
		gfx_frame_end(gfx);
		game_readkey();

		lords[i].is_dead = TRUE;
		return 0;
	}

	
	clear_keybuf();

	ticks = 0;
	while(!done) {
		if (game_mode != MODE_HOTSEAT) game_poll();
		if (game_mode == MODE_SLAVE && time_to_watch) break;
		if (lost_connection) return 2;

		while (ticks)
		{	
			ticks--;
			if (key[KEY_LCONTROL] || key[KEY_RCONTROL])
			{
				while (key[KEY_LCONTROL] || key[KEY_RCONTROL]) rest(1);
				done = 1;
				ticks = 0;
			}

			if (close_button_pressed)
			{
				simulate_keypress(KEY_ESC << 8);
			}

			if (keypressed())
			{
				int scancode = readkey(), ascii = scancode & 0xff, k = scancode >> 8;
								
				if (k == KEY_ESC)
				{
					char response[80];
					BITMAP *temp = create_bitmap(320,200);
					if (!temp) exit(1);
					clear(temp);
					map_draw(map, temp, 8,8);
					response[0] = 0;
					prompt(temp, response, 2, 8,160,15,0,"REALLY QUIT (Y/N)?");
					destroy_bitmap(temp);
					if (response[0] == 'Y') return 2;
					close_button_pressed = FALSE;
				}
				else if (k >= KEY_A && k <= KEY_Z && game_mode != MODE_HOTSEAT)
				{
					game_type_chat_message(i, ascii);
				}

				if (mode == MODE_LOOKING)
				{
					if (k != KEY_SPACE) no_troops_msg = 0;
					
					if (k == KEY_LEFT && x)
						x--;
					else if (k == KEY_RIGHT && x < 18)
						x++;
					else if (k == KEY_UP && y)
						y--;
					else if (k == KEY_DOWN && y < 18)
						y++;
					else if (k == KEY_SPACE)
					{
						/* check if sending from a castle */
						CASTLE *castle = map->grid[y][x].castle; 
						ARMY *army = army_get_idle(map->armies, x,y);
						if ((castle && castle->player == i && castle->soldiers) || (army && army->player == i))
						{
							max_soldiers = !castle ? army->size : castle->soldiers;
							mode = MODE_CREATE_ARMY;
							sending = 0;
						}
						else
							no_troops_msg = 1 - no_troops_msg;
					}

					castle = map->grid[y][x].castle;
				}
				else if (mode == MODE_CREATE_ARMY)
				{
				
					if (k == KEY_LEFT)
						sending--;
					else if (k == KEY_RIGHT)
						sending++;
					else if (k == KEY_UP)
						sending+=10;
					else if (k == KEY_DOWN)
						sending-=10;
					else if (k == KEY_SPACE)
					{
						if (!sending)
							mode = MODE_LOOKING;
						else
						{
							mode = MODE_SENDING;
							sx = x; sy = y;
						}
					}
					
					if (sending < 0) sending = 0;
					else if (sending > max_soldiers) sending = max_soldiers;
				}
				else if (mode == MODE_SENDING)
				{
					if (k == KEY_LEFT && sx)
						sx--;
					else if (k == KEY_RIGHT && sx < 18)
						sx++;
					else if (k == KEY_UP && sy)
						sy--;
					else if (k == KEY_DOWN && sy < 18)
						sy++;
					else if (k == KEY_SPACE)
					{
						ARMY *army = army_add(&map->armies);
						CASTLE *castle = map->grid[y][x].castle; 
						if (!castle)
						{
							ARMY *idle_army = army_get_idle(map->armies, x,y);
							idle_army->size -= sending;
							if (idle_army->size == 0)
								army_remove(&map->armies, idle_army);
						}
						else
							castle->soldiers -= sending;
					
						army->player = i;
						army->x = x;
						army->y = y;
						army->dx = sx;
						army->dy = sy;
						army->size = sending;
						army->idle = FALSE;
						army->moves_left = 5;

						mode = MODE_SEND_CONFIRM;

						if (game_mode == MODE_SLAVE)
						{
							/* report to master */
							PACKET_BOX *pb = packet_box_create("ASGN");
							ENetPacket *packet;

							packet_box_add_int(pb, x, 8);
							packet_box_add_int(pb, y, 8);
							packet_box_add_int(pb, sx, 8);
							packet_box_add_int(pb, sy, 8);
							packet_box_add_int(pb, sending, 8);
							packet_box_serialize(pb);

							packet = enet_packet_create(pb->data, pb->size, ENET_PACKET_FLAG_RELIABLE);
							enet_peer_send(peer, 1, packet);

							packet_box_destroy(pb);
						}
					}
				}
				else if (mode == MODE_SEND_CONFIRM)
				{

					if (k == KEY_SPACE) mode = MODE_LOOKING;
				}
			}
		}

		buffer = gfx_frame_start(gfx);
		map_draw(map, buffer, 8,8);

		if (no_troops_msg) 
			textprintf_ex(buffer, font64, 8,0, 4+i,0,"NO TROOPS AVAILABLE-HIT SPACE");
		else if (mode == MODE_LOOKING)
			textprintf_ex(buffer, font64, 8,0, 4+i,0,"%s LOOKING (CTRL TO END)", lords[i].name);
		else if (mode == MODE_CREATE_ARMY)
			textprintf_ex(buffer, font64, 8,0, 4+i,0,"ARROW KEYS SIZE, SPACE SEND");
		else if (mode == MODE_SENDING)
			textprintf_ex(buffer, font64, 8,0, 4+i,0,"POSITION X WITH ARROW KEYS, THEN SPACE");
		else if (mode == MODE_SEND_CONFIRM)
			textprintf_ex(buffer, font64, 8,0, 4+i,0,"HIT SPACE TO CONTINUE");

		army_is_here = army_check_spot(map->armies, x, y);

		switch (map->grid[y][x].tile)
		{
			case TILE_GRASS:
				if (army_is_here == -1)
					textprintf_ex(buffer, font64, 8,20*8, map->color[TILE_GRASS],0, "GRASS");
				else if (army_is_here != i)
					textprintf_ex(buffer, font64, 8,20*8,4+army_is_here,0, "%s'S TROOPS", lords[army_is_here].name);
				
				break;
			case TILE_TREES:
				textprintf_ex(buffer, font64, 8,20*8, map->color[TILE_TREES],0, "WOODS");
				break;
			case TILE_ROCKS:
				textprintf_ex(buffer, font64, 8,20*8, map->color[TILE_ROCKS],0, "ROCKS");
				break;
			case TILE_CASTLE:
			{
				CASTLE *castle = map->grid[y][x].castle; 
				int player = castle->player;
				if (player != i)
					textprintf_ex(buffer, font64, 8,20*8, 4+player, 0, "%s'S CASTLE", lords[player].name);
				else
				{
					if (mode == MODE_LOOKING)
					{
						textprintf_ex(buffer, font64, 8,20*8, 4+player, 0, "   YOUR CASTLE");
						textprintf_ex(buffer, font64, 8,21*8, 4+player, 0, "TROOPS: %d  LEVY: %d     %s", castle->soldiers, castle->levy, castle->soldiers ? "(SPACE-SEND)" : "");
					}
				}
				break;
			}
		}

		if (mode == MODE_CREATE_ARMY)
		{
			textprintf_ex(buffer, font64, 8,20*8, 4+i, 0, "TROOPS 1-%d", max_soldiers);
			textprintf_ex(buffer, font64, 8,21*8, 4+i, 0, "  SENDING %d", sending);
		}
		else if (army_is_here == i && mode != MODE_SEND_CONFIRM)
		{
			if (map->grid[y][x].tile != TILE_CASTLE)
			{
				ARMY *idle = army_get_idle(map->armies, x,y);
				int color = 4+i;

				if (map->grid[y][x].tile == TILE_TREES)
				{
					color = map->color[TILE_TREES];
					textprintf_ex(buffer, font64, 8,20*8, color,0, "    YOUR TROOPS, HIDDEN");
				}
				else
					textprintf_ex(buffer, font64, 8,20*8, color,0, "    YOUR TROOPS");

				textprintf_ex(buffer, font64, 8,21*8, color,0, "IDLE TROOPS: %d", idle ? idle->size : 0);
				if (idle) textprintf_ex(buffer, font64, 208,21*8, color,0, "(SPACE-SEND)");					
			}
			
			army_print(map->armies, x, y, buffer, 32,22*8);
		}

		/* draw cursor */
		rect(buffer, 8+ x*16,8+y*8, 8+ x*16+15,8+y*8+7, 9);
		rect(buffer, 8+ x*16-1,8+y*8, 8+ x*16+16,8+y*8+7, 9);
		rect(buffer, 8+ x*16-2,8+y*8-1, 8+ x*16+17,8+y*8+8, 9);

		if (mode == MODE_SENDING)
		{
			/* draw X cursor. Why didn't I use a sprite??? */
			hline(buffer, 8+sx*16+2, 8+sy*8-1, 8+sx*16+13, 9);
			rect(buffer, 8+sx*16+4,8+sy*8, 8+sx*16+11,8+sy*8+1, 9);
			putpixel(buffer, 8+sx*16+4,8+sy*8+2, 9);
			putpixel(buffer, 8+sx*16+5,8+sy*8+2, 9);
			putpixel(buffer, 8+sx*16+10,8+sy*8+2, 9);
			putpixel(buffer, 8+sx*16+11,8+sy*8+2, 9);
			
			rectfill(buffer, 8+sx*16,8+sy*8+2, 8+sx*16+3, 8+sy*8+5, 9);
			rectfill(buffer, 8+sx*16+12,8+sy*8+2, 8+sx*16+15, 8+sy*8+5, 9);
			rectfill(buffer, 8+sx*16-2,8+sy*8+1,  8+sx*16-1,8+sy*8+6, 9);
			rectfill(buffer, 8+sx*16+16,8+sy*8+1,  8+sx*16+17,8+sy*8+6, 9);
			
			putpixel(buffer, 8+sx*16+4,8+sy*8+5, 9);
			putpixel(buffer, 8+sx*16+5,8+sy*8+5, 9);
			putpixel(buffer, 8+sx*16+10,8+sy*8+5, 9);
			putpixel(buffer, 8+sx*16+11,8+sy*8+5, 9);

			rect(buffer, 8+sx*16+4,8+sy*8+6, 8+sx*16+11,8+sy*8+7, 9);
			hline(buffer, 8+sx*16+2, 8+sy*8+8, 8+sx*16+13, 9);
		}
		else if (mode == MODE_SEND_CONFIRM)
		{
			textprintf_ex(buffer, font64, 8,20*8, 4+i, 0, "ARMY OF %d SENT", sending);
		}

		gfx_frame_end(gfx);

		rest(10);
	}

	return 1;
}

int game_take_turns()
{
	switch (game_mode)
	{
		case MODE_HOTSEAT:
		{	
			int i;
			for (i=1; i<=lord_count; i++)
			{
				if (game_take_turn(i) == 2) return TRUE;
			}
			break;
		}	

		case MODE_MASTER:
		{
			game_send_soldier_count(NULL);
			if (map->armies) game_send_army_count();			
			game_send_start_signal();

			if (game_take_turn(1) == 2) return TRUE;
			lords[1].done_with_turn = TRUE;
			game_master_broadcast_done(1);
			still_moving--;
			break;
		}

		case MODE_SLAVE:
		{
			int whatever; /* 4 hours until monday noon */

			while (!key[KEY_ESC] && !i_can_move) game_poll();
			if (!i_can_move) return TRUE;
			i_can_move = FALSE;

			whatever = game_take_turn(who_i_am);
			game_send_done_with_turn_signal();
			
			if (whatever == 2)
				return TRUE;
	
			break;
		}
	}

	return FALSE;
}


/* Process the movements. This is valid for hotseat and master modes. */
void game_move()
{
	ARMY *army = map->armies;
	CASTLE *castle = map->castles;
	BITMAP *buffer;
	int all_done = FALSE;

	if (game_mode == MODE_MASTER)
	{
		while (!key[KEY_ESC] && still_moving)
		{
			buffer = gfx_frame_start(gfx);
			textprintf_ex(buffer, font64, 8,0, 15,0, "WAITING FOR OTHERS TO FINISH");
			map_draw(map, buffer, 8,8);
			gfx_frame_end(gfx);

			game_poll();
		}
	}

	buffer = gfx_frame_start(gfx);
	textprintf_ex(buffer, font64, 8,0, 15,0, "HIT SPACE TO SEE ARMIES MARCH");
	map_draw(map, buffer, 8,8);
	gfx_frame_end(gfx);
	game_pause();

	
	if (game_mode == MODE_MASTER)
		game_send_watch_signal();


	while (army)
	{
		army->moves_left = 5;
		army = army->next;
	}

	play_sample(move, 255, 128, 1000, 1);

	while(!all_done)
	{
		all_done = TRUE;
		if (lost_connection) return;
		
		army = map->armies;
		while (army)
		{
			while (ticks < 20)
			{
				game_poll();
			}
			ticks = 0;

			if (!army->idle && army->moves_left > 0)
			{
				int enemy;
				
				MAP_POINT *next = map_next_step(army->player, &map->grid[army->y][army->x], &map->grid[army->dy][army->dx]);
				if (next)
				{
					army->x = next->x;
					army->y = next->y;

					if (army->x == army->dx && army->y == army->dy) army->idle = TRUE;					
					if (game_mode == MODE_MASTER) game_send_army_position(army);
				}
				else
				{
					army->idle = TRUE;
					army->moves_left = 0;
				}

				if ((enemy = map_get_enemy(map, army->x, army->y, army->player)) != -1)
				{
					/* A FIGHT! */
					float attack_bonus = 1.0, defense_bonus = 1.0;
					int first_attack = TRUE;
					int pan = 0;
					int enemy_size = map_get_enemy_size(map, army->x, army->y, army->player);
					int initial_enemy_size = enemy_size;

					switch (map->grid[army->y][army->x].tile)
					{
						case TILE_GRASS:
							attack_bonus = 1.10;
							break;

						case TILE_TREES:
							defense_bonus = 1.10;
							break;

						case TILE_CASTLE:
							attack_bonus = 0.90;							
							break;
					}

					/* CPU is easier to defeat, but harder to defend */
					if (army->player == 0) attack_bonus += 0.3;
					if (enemy == 0) defense_bonus -= 0.3;					
					
					if (game_mode == MODE_MASTER)
					{
						game_send_attack_signal(army, enemy, enemy_size);
					}

					buffer = gfx_frame_start(gfx);
					textprintf_ex(buffer, font64, 8,0, 3,0, "HIT SPACE TO SEE OUTCOME");
					map_draw(map, buffer, 8,8);
					textprintf_ex(buffer, font64, 24,160, 4+army->player,0, "%s ATTACKS ", lords[army->player].name);
					textprintf_ex(buffer, font64, 96+strlen(lords[army->player].name) * 8,160, 4+enemy,0, "%s", lords[enemy].name);
					textprintf_ex(buffer, font64, 8,168, 3,0, "        %d              %d", army->size, enemy_size);
					draw_sprite(buffer, attack_sprite[0], army->x*16 + 8 - 2, army->y*8 + 8 - 1);
					gfx_frame_end(gfx);
					game_pause();					
					
					do
					{
						int dead = ((rand() % (int)((army->size / (first_attack ? 10.0 : 4.0) + 5.0))) + 1) * attack_bonus;

						while (ticks < 30) game_poll();
						ticks = 0;

						enemy_size -= dead;
						if (enemy_size <= 0) enemy_size = 0;
						else
						{
							dead = (((rand() % (int)((enemy_size / 4.0) + 5.0))) + 1) * defense_bonus;
							army->size -= dead;
							if (army->size < 0) army->size = 0;
						}

						if (enemy_size && army->size) play_sample(poof, 140, pan = 255-pan, 1000, 0);

						buffer = gfx_frame_start(gfx);
						//textprintf_ex(buffer, font64, 8,0, 3,0, "HIT SPACE TO SEE OUTCOME");
						map_draw(map, buffer, 8,8);
						textprintf_ex(buffer, font64, 24,160, 4+army->player,0, "%s ATTACKS ", lords[army->player].name);
						textprintf_ex(buffer, font64, 96+strlen(lords[army->player].name) * 8,160, 4+enemy,0, "%s", lords[enemy].name);
						textprintf_ex(buffer, font64, 8,168, 3,0, "        %d              %d", army->size, enemy_size);
						draw_sprite(buffer, attack_sprite[army->size % 3], army->x*16 + 8 - 2, army->y*8 + 8 - 1);
						gfx_frame_end(gfx);

						if (game_mode == MODE_MASTER)
							game_send_attack_signal(army, enemy, enemy_size);
					} while (enemy_size > 0 && army->size > 0);

					/* reduce defense troops */
					if (initial_enemy_size > enemy_size)
						map_reduce_troops(map, army->x, army->y, enemy, initial_enemy_size - enemy_size);

					if (!enemy_size && map->grid[army->y][army->x].castle)
					{
						/* castle was taken over */
						play_sample(ahahaa, 50, 128, 1000, 0);
						map->grid[army->y][army->x].castle->soldiers = 0;
						map->grid[army->y][army->x].castle->player = army->player;

						if (game_mode == MODE_MASTER) game_send_soldier_count(map->grid[army->y][army->x].castle);
					}					

					if (game_mode == MODE_MASTER)
					{
						game_send_attack_over(enemy_size);
					}

					buffer = gfx_frame_start(gfx);
					textprintf_ex(buffer, font64, 8,0, 3,0, "HIT SPACE TO CONTINUE");
					map_draw(map, buffer, 8,8);
					if (!army->size)
					{
						/* defenders win */
						textprintf_ex(buffer, font64, 24,160, 4+enemy,0, "%s REPELLS INVADERS", lords[enemy].name);
						gfx_frame_end(gfx);
						game_pause();
						if (game_mode == MODE_MASTER) 
						{
							game_send_watch_signal();
							game_send_army_dead(army);
						}
						army = army_remove(&map->armies, army);
						continue;
					}
					else
					{
						/* attackers win */
						textprintf_ex(buffer, font64, 24,160, 4+army->player,0, "%s IS VICTORIOUS", lords[army->player].name);
						gfx_frame_end(gfx);
						game_pause();
						if (game_mode == MODE_MASTER) game_send_watch_signal();
					}
				}				

				if (--army->moves_left > 0) all_done = FALSE;
			}
			army = army->next;

			buffer = gfx_frame_start(gfx);
			textprintf_ex(buffer, font64, 8,0, 15,0, "MOVING ARMIES...");
			map_draw(map, buffer, 8,8);
			gfx_frame_end(gfx);

		}	
	}

	stop_sample(move);


	/* merge any idle armies together */
	army = map->armies;
	while (army)
	{
		if (army->idle)
		{			
			if (map->grid[army->y][army->x].castle)
			{
				/* let's place this troop inside the castle */				
				map->grid[army->y][army->x].castle->soldiers += army->size;
				if (game_mode == MODE_MASTER) game_send_army_dead(army);
				army = army_remove(&map->armies, army); /* returns the next army */				
				continue;
			}
			else
			{
				/* we can start with the next army, because previous ones have already been checked */
				ARMY *army2 = army->next;
				
				while (army2)
				{
					if (army2->idle && army2->x == army->x && army2->y == army->y)
					{
						/* two idle armies on same title */
						if (army2 == army->next)
						{
							/* gotta do this, or else army's next pointer is pointing to junk */
							army->next = army2->next;
						}
						army->size += army2->size;
						if (game_mode == MODE_MASTER) game_send_army_dead(army2);
						army_remove(&map->armies, army2);
					}				
					army2 = army2->next;
				}
			}
		}

		army = army->next;
	}

	/* increase castle strength */
	while(castle)
	{
		castle->soldiers += castle->levy;
		castle = castle->next;
	}
}


void game_watch_move()
{
	/*
		If a turn is REALLY fast on a lagged connection, the player might miss the entire thing.
		So we need to check the "i_can_move" flag as well. If that becomes TRUE, then the watching
		is effectively over.
	*/
	while (!key[KEY_ESC] && !time_to_watch && !i_can_move)
	{
		BITMAP *buffer = gfx_frame_start(gfx);
		textprintf_ex(buffer, font64, 8,0, 15,0, "WAITING FOR OTHERS TO FINISH");
		map_draw(map, buffer, 8,8);
		gfx_frame_end(gfx);

		game_poll();

		if (lost_connection) return;
	}
	if (!time_to_watch && !i_can_move) exit(1);
 
	watch_status.mode = WATCH_MOVING;

	play_sample(move, 255, 128, 1000, 1);

	while (!key[KEY_ESC] && time_to_watch)
	{
		BITMAP *buffer = gfx_frame_start(gfx);	
		map_draw(map, buffer, 8,8);
		switch (watch_status.mode)
		{
			case WATCH_MOVING:
				textprintf_ex(buffer, font64, 8,0, 15,0, "MOVING ARMIES...");
				break;

			case WATCH_FIGHTING:
				textprintf_ex(buffer, font64, 24,160, 4+watch_status.attacker_id ,0, "%s ATTACKS ", lords[watch_status.attacker_id].name);
				textprintf_ex(buffer, font64, 96+strlen(lords[watch_status.attacker_id].name) * 8,160, 4+watch_status.defender_id,0, "%s", lords[watch_status.defender_id].name);
				textprintf_ex(buffer, font64, 8,168, 3,0, "        %d              %d", watch_status.attacker_size , watch_status.defender_size );
				draw_sprite(buffer, attack_sprite[watch_status.attacker_size % 3], watch_status.x*16 + 8 - 2, watch_status.y*8 + 8 - 1);

				break;

			case WATCH_END_OF_FIGHT:
				if (watch_status.defender_size)
					textprintf_ex(buffer, font64, 24,160, 4+watch_status.defender_id,0, "%s REPELLS INVADERS", lords[watch_status.defender_id].name);
				else
					textprintf_ex(buffer, font64, 24,160, 4+watch_status.attacker_id,0, "%s IS VICTORIOUS", lords[watch_status.attacker_id].name);
				break;
		}

		gfx_frame_end(gfx);
		game_poll();

		if (lost_connection) return;
	}

	stop_sample(move);
}


/* this function needs to be called all over the place */
void game_poll()
{
	if (game_mode != MODE_HOTSEAT)
	{
		static int key_plus = FALSE;
		static int key_minus = FALSE;
		int i, ts = time(NULL);

		for (i=1; i<=lord_count; i++)
		{
			if (lords[i].is_dead == TRUE)
			{
				textprintf_ex(activity_bmp, font64, 0,(i-1)*8,i+4, 0, "-");					
			}
			else
			{
				textprintf_ex(activity_bmp, font64, 0,(i-1)*8,0,i+4, "%c", 
					lords[i].done_with_turn == TRUE ? ' ' :
					(lords[i].last_activity + 1 > ts ? '*' : '-')
				);
			}
		}

		if (chatbox)
		{
			if (key[KEY_PLUS_PAD])
			{
				if (!key_plus)
				{
					key_plus = TRUE;
					chatbox_next();					
				}
			}
			else key_plus = FALSE;
			
			if (key[KEY_MINUS_PAD])
			{
				if (!key_minus)
				{
					key_minus = TRUE;
					chatbox_previous();
				}
			}
			else key_minus = FALSE;

			chatbox_update();
		}

		
		game_poll_enet();
	}
	else
		rest(10);
}